 //////////////////////////////////////////////////////////////////////////////
//  Copyright 2014 by Autodesk, Inc.  All rights reserved.
//
// This computer source code and related instructions and comments 
// are the unpublished confidential and proprietary information of 
// Autodesk, Inc. and are protected under applicable copyright and 
// trade secret law.  They may not be disclosed to, copied or used 
// by any third party without the prior written consent of Autodesk, Inc.
//////////////////////////////////////////////////////////////////////////////
 
/*globals $,jQuery,document, window, navigator*/

/*!
 * ntp.utils - Utility functions
 */

// Namespace
var ntp = ntp || {};
ntp.utils = ntp.utils || {};
 
/**
 * Creates a callback function that calls the function 'name' on 'obj'.
 */
ntp.utils.makeCallback = function (obj, name) {
    return function () {
        var func = obj[name];
        if (func) {
            return func.apply(obj, arguments);
        }
    };
};

/**
 * Create a callback function that calls the function 'name' on 'obj' with fixed arguments
 */
ntp.utils.makeCallbackWithArgs = function (obj, name, args) {
    var funcArgs = Array.prototype.slice.call(arguments, 2);
    return function () {
        var func = obj[name];
        if (func) {
            return func.apply(obj, funcArgs);
        }
    };
};

/**
 * Gets a localized string with text-id, or apply localization on an element. 
 */
ntp.utils.localize = (function () {
    return function (param, param2) {
        if (typeof param === 'string') {
            if (ntp.strings) {
                return ntp.strings[param];
            }

            $.error('Localized resource has not been loaded.');
        } else if (param instanceof jQuery && typeof param2 === 'string') {
            if (ntp.strings) {
                param.html(ntp.strings[param2]);
            } else {
                param.attr('data-text-id', param2);
            }
        } else {
            $.error('Invalid parameter.');
        }
    };
})();

/**
 * Adds a CSS style rule to the page
 */
ntp.utils.addRule = (function () {
    var styleSheet = null;
    
    return function (rule) {
        if (typeof rule !== 'string') {
            $.error('Invalid parameter.');
            return;
        }
        if (styleSheet === null) {
            var style = document.createElement('style');
            document.getElementsByTagName('head')[0].appendChild(style);
            if (!window.createPopup) {
               style.appendChild(document.createTextNode(''));
            }
            styleSheet = document.styleSheets[document.styleSheets.length - 1];
        }
        styleSheet.insertRule(rule, styleSheet.cssRules.length);
    };
})();

/**
 * Resolves the absolute URL for a relative path
 */
ntp.utils.resolvePath = (function () {
    var aElement = null;
    
    return function (url) {
        if (typeof url !== 'string') {
            $.error('Invalid parameter.');
            return;
        }
        if (aElement === null) {
            aElement = document.createElement('a');
        }
        aElement.href = url;
        return aElement.href;
    };
})();

/**
 * A utility for executing function on timeout. If the same function
 * is scheduled again before the previous one is executed, the previous
 * timeout would be cleared.
 */
ntp.utils.timeout = (function () {
    // A map from function to timeout handle
    var timeoutHandles = {};
    
    return function (func, time) {
        if (typeof func !== 'function') {
            return; 
        }
        
        var timeoutHandle = timeoutHandles[func];
        if (timeoutHandle) {
            window.clearTimeout(timeoutHandle);
        };
        
        timeoutHandles[func] = window.setTimeout(function () {
            func();
            delete timeoutHandles[func];
        }, time);
    };
})();

/**
 * Utilities for extracting info from the query string.
 *
 * The product, locale, and automatic settings are passed in the URL, e.g. 
 *     index.html?lang=zh-CS&prod=ACAD&setting=%7B%0A%20%22openSheetSet%22%20%3A%20%22disabled%22%0A%7D
 */
ntp.utils.query = {
    
    locale: function () {
        var results = /[\\?&]lang=([^&#]*)/.exec(window.location.search);
        return results !== null && results !== '' ? results[1] : 'en-US';
    },
    
    product: function () {
        var results = /[\\?&]prod=([^&#]*)/.exec(window.location.search);
        return results !== null && results !== '' ? results[1] : null;
    },
    
    settings: function () {
        var results = /[\\?&]settings=([^&#]*)/.exec(window.location.search);
        if (results === null || results === '') {
            return null;
        }
        try {
            var settingsJSON = decodeURIComponent(results[1].replace(/\+/g, ' '));
            return $.parseJSON(settingsJSON);
        } catch (e) {
            // Failed to parse the JSON object
            return null;
        }
    }
};

/**
 * Shows the leaving Autodesk notice in a lightbox. Opens the URL in
 * external browser if user confirms.
 */
ntp.utils.openExternalUrlWithBrowser = (function () {
    
    var $leaveLightBox = null;
    var handleContinueButton = function (event) {
        if (event.keyCode == 13 && $leaveLightBox.lightbox('isShowing')) {
            if (!$('#button_leave_cancel').is(':focus')) {
                event.preventDefault();
                $('#button_leave_continue').click();
            }
        }
    };
    
    return function (url) {
        if ($leaveLightBox == null) {
            $leaveLightBox = $('#leave_lightbox');
            
            $leaveLightBox.lightbox('bind', 'lightboxshown', function () {
                $(document).on('keydown', handleContinueButton);
            });
            
            $leaveLightBox.lightbox('bind', 'lightboxhidden', function () {
                $(document).off('keydown', handleContinueButton);
            });
        }
        $('#button_leave_continue').off('click').click(function () {
            ntp.acad.openUrlWithBrowser(url);
            $leaveLightBox.lightbox('hide');
        });
        
        $('#button_leave_cancel').off('click').click(function () {
            $leaveLightBox.lightbox('hide');
        });
        
        $leaveLightBox.lightbox();
    };
})();

ntp.utils.string = {
        format : function() {
            if( arguments.length == 0 )
                return null;

            var str = arguments[0]; 
            for(var i=1;i<arguments.length;i++) {
                var re = new RegExp('\\{' + (i-1) + '\\}','gm');
                str = str.replace(re, arguments[i]);
            }
            return str;
        }
};
// SIG // Begin signature block
// SIG // MIIZ7QYJKoZIhvcNAQcCoIIZ3jCCGdoCAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // DlwcuCMvQ2iE2Hon98EFNx2Krp3TwKr2nKqwXhmdHiig
// SIG // ggpjMIIFAjCCA+qgAwIBAgIQZLyxx87k2lhp1Y8eIvvp
// SIG // 9DANBgkqhkiG9w0BAQsFADB/MQswCQYDVQQGEwJVUzEd
// SIG // MBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xHzAd
// SIG // BgNVBAsTFlN5bWFudGVjIFRydXN0IE5ldHdvcmsxMDAu
// SIG // BgNVBAMTJ1N5bWFudGVjIENsYXNzIDMgU0hBMjU2IENv
// SIG // ZGUgU2lnbmluZyBDQTAeFw0xNTA5MDMwMDAwMDBaFw0x
// SIG // NjA5MDIyMzU5NTlaMIGIMQswCQYDVQQGEwJVUzETMBEG
// SIG // A1UECAwKQ2FsaWZvcm5pYTETMBEGA1UEBwwKU2FuIFJh
// SIG // ZmFlbDEWMBQGA1UECgwNQXV0b2Rlc2ssIEluYzEfMB0G
// SIG // A1UECwwWRGVzaWduIFNvbHV0aW9ucyBHcm91cDEWMBQG
// SIG // A1UEAwwNQXV0b2Rlc2ssIEluYzCCASIwDQYJKoZIhvcN
// SIG // AQEBBQADggEPADCCAQoCggEBAOqZ9OjPzARqd9dP4fq2
// SIG // qXdpTJolFYJ8Wxg8WcZVciX3l+dljew4R9C8WR/0tW17
// SIG // lLANaSORF4IYmWTgdpFg5PGR3s+R4A461hpRIG9QthGU
// SIG // q8Bt7af5VbCjjKtYBNq/x3ukqVsw/1/qljGVplDOyr97
// SIG // 6ktnCm0/wL0N+ubk5WAIgv+I9E0i6+GX1hRm+eRliYKe
// SIG // KF0/gBLyDh3ut+N3HNPrnfjfec7q2Czd82CeRBCToR1E
// SIG // 7Cp6yBBdGUfb3JE0PnE1+6n0qJnsvs2bsCIzqVo3A0iM
// SIG // HBsbioxa5DYyXuEfNCktosUlFXfegd514ZJns6YDr4Pl
// SIG // ahC0+lK3hgJy5vkCAwEAAaOCAW4wggFqMAkGA1UdEwQC
// SIG // MAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsG
// SIG // AQUFBwMDMGYGA1UdIARfMF0wWwYLYIZIAYb4RQEHFwMw
// SIG // TDAjBggrBgEFBQcCARYXaHR0cHM6Ly9kLnN5bWNiLmNv
// SIG // bS9jcHMwJQYIKwYBBQUHAgIwGRoXaHR0cHM6Ly9kLnN5
// SIG // bWNiLmNvbS9ycGEwHwYDVR0jBBgwFoAUljtT8Hkzl699
// SIG // g+8uK8zKt4YecmYwKwYDVR0fBCQwIjAgoB6gHIYaaHR0
// SIG // cDovL3N2LnN5bWNiLmNvbS9zdi5jcmwwVwYIKwYBBQUH
// SIG // AQEESzBJMB8GCCsGAQUFBzABhhNodHRwOi8vc3Yuc3lt
// SIG // Y2QuY29tMCYGCCsGAQUFBzAChhpodHRwOi8vc3Yuc3lt
// SIG // Y2IuY29tL3N2LmNydDARBglghkgBhvhCAQEEBAMCBBAw
// SIG // FgYKKwYBBAGCNwIBGwQIMAYBAQABAf8wDQYJKoZIhvcN
// SIG // AQELBQADggEBAB6BYdY8nzLWS3nsk/xM5CWdizR4wyDn
// SIG // 8wfGVbcuEq/88byNnY0ODxisa7AELvLe6b48DorFFP2n
// SIG // Hw0qOeIxfg2w+1DzWzYQhMfLi4V/Pp1OmCYvYEsTb/oU
// SIG // +YbiSgSFjrN5Pi2LN5Oxw5Ey7WAK2vhWr4EL/7QDHoso
// SIG // 4a3lS3WklPz2K/Nlhk4i8S230BlGCHsZGDY7PtfowWdz
// SIG // cq50xCEIScERSRtLd+YL6SWXk+6gX9HVcFxtvLQ4N9Df
// SIG // F/BKk13pYgyvZPkedIKA4yYCy4/VjouNoKhgM/aaGLAy
// SIG // Dv3ujRdXLCcosR4MbD8X1v9z6WkgG5YZZicWPPsJN7ty
// SIG // B1wwggVZMIIEQaADAgECAhA9eNf5dklgsmF99PAeyoYq
// SIG // MA0GCSqGSIb3DQEBCwUAMIHKMQswCQYDVQQGEwJVUzEX
// SIG // MBUGA1UEChMOVmVyaVNpZ24sIEluYy4xHzAdBgNVBAsT
// SIG // FlZlcmlTaWduIFRydXN0IE5ldHdvcmsxOjA4BgNVBAsT
// SIG // MShjKSAyMDA2IFZlcmlTaWduLCBJbmMuIC0gRm9yIGF1
// SIG // dGhvcml6ZWQgdXNlIG9ubHkxRTBDBgNVBAMTPFZlcmlT
// SIG // aWduIENsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlm
// SIG // aWNhdGlvbiBBdXRob3JpdHkgLSBHNTAeFw0xMzEyMTAw
// SIG // MDAwMDBaFw0yMzEyMDkyMzU5NTlaMH8xCzAJBgNVBAYT
// SIG // AlVTMR0wGwYDVQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlv
// SIG // bjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29y
// SIG // azEwMC4GA1UEAxMnU3ltYW50ZWMgQ2xhc3MgMyBTSEEy
// SIG // NTYgQ29kZSBTaWduaW5nIENBMIIBIjANBgkqhkiG9w0B
// SIG // AQEFAAOCAQ8AMIIBCgKCAQEAl4MeABavLLHSCMTXaJNR
// SIG // YB5x9uJHtNtYTSNiarS/WhtR96MNGHdou9g2qy8hUNqe
// SIG // 8+dfJ04LwpfICXCTqdpcDU6kDZGgtOwUzpFyVC7Oo9tE
// SIG // 6VIbP0E8ykrkqsDoOatTzCHQzM9/m+bCzFhqghXuPTbP
// SIG // HMWXBySO8Xu+MS09bty1mUKfS2GVXxxw7hd924vlYYl4
// SIG // x2gbrxF4GpiuxFVHU9mzMtahDkZAxZeSitFTp5lbhTVX
// SIG // 0+qTYmEgCscwdyQRTWKDtrp7aIIx7mXK3/nVjbI13Iwr
// SIG // b2pyXGCEnPIMlF7AVlIASMzT+KV93i/XE+Q4qITVRrgT
// SIG // hsIbnepaON2b2wIDAQABo4IBgzCCAX8wLwYIKwYBBQUH
// SIG // AQEEIzAhMB8GCCsGAQUFBzABhhNodHRwOi8vczIuc3lt
// SIG // Y2IuY29tMBIGA1UdEwEB/wQIMAYBAf8CAQAwbAYDVR0g
// SIG // BGUwYzBhBgtghkgBhvhFAQcXAzBSMCYGCCsGAQUFBwIB
// SIG // FhpodHRwOi8vd3d3LnN5bWF1dGguY29tL2NwczAoBggr
// SIG // BgEFBQcCAjAcGhpodHRwOi8vd3d3LnN5bWF1dGguY29t
// SIG // L3JwYTAwBgNVHR8EKTAnMCWgI6Ahhh9odHRwOi8vczEu
// SIG // c3ltY2IuY29tL3BjYTMtZzUuY3JsMB0GA1UdJQQWMBQG
// SIG // CCsGAQUFBwMCBggrBgEFBQcDAzAOBgNVHQ8BAf8EBAMC
// SIG // AQYwKQYDVR0RBCIwIKQeMBwxGjAYBgNVBAMTEVN5bWFu
// SIG // dGVjUEtJLTEtNTY3MB0GA1UdDgQWBBSWO1PweTOXr32D
// SIG // 7y4rzMq3hh5yZjAfBgNVHSMEGDAWgBR/02Wnwt3su/Aw
// SIG // CfNDOfoCrzMxMzANBgkqhkiG9w0BAQsFAAOCAQEAE4Ua
// SIG // HmmpN/egvaSvfh1hU/6djF4MpnUeeBcj3f3sGgNVOftx
// SIG // lcdlWqeOMNJEWmHbcG/aIQXCLnO6SfHRk/5dyc1eA+CJ
// SIG // nj90Htf3OIup1s+7NS8zWKiSVtHITTuC5nmEFvwosLFH
// SIG // 8x2iPu6H2aZ/pFalP62ELinefLyoqqM9BAHqupOiDlAi
// SIG // KRdMh+Q6EV/WpCWJmwVrL7TJAUwnewusGQUioGAVP9rJ
// SIG // +01Mj/tyZ3f9J5THujUOiEn+jf0or0oSvQ2zlwXeRAwV
// SIG // +jYrA9zBUAHxoRFdFOXivSdLVL4rhF4PpsN0BQrvl8OJ
// SIG // IrEfd/O9zUPU8UypP7WLhK9k8tAUITGCDuIwgg7eAgEB
// SIG // MIGTMH8xCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1h
// SIG // bnRlYyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50
// SIG // ZWMgVHJ1c3QgTmV0d29yazEwMC4GA1UEAxMnU3ltYW50
// SIG // ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
// SIG // AhBkvLHHzuTaWGnVjx4i++n0MA0GCWCGSAFlAwQCAQUA
// SIG // oHwwEAYKKwYBBAGCNwIBDDECMAAwGQYJKoZIhvcNAQkD
// SIG // MQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwG
// SIG // CisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIGFV3gur
// SIG // wfxNuWJe28U4uJuwgzZm4yysDuX75zIY2m6lMA0GCSqG
// SIG // SIb3DQEBAQUABIIBALtNPzGem0o3htieCOouboQDBsHl
// SIG // YUfxhVOrHfPhMxBq3N8tY5SqRHRImyMJS2zQR6fiJmfC
// SIG // zm2SEpLLLicsPKuix4caqxIvo2Sq1TB/sghb7dbB+8JN
// SIG // DMR4to6OzNPNNZ5als9i1xSvSYi5PkWsu3BiJAAhr4Z3
// SIG // 5jl9A5+3rEhBxiEaw1SLt9lFCo4w9Nsy5T2j2DWlaKcF
// SIG // /j0WNDtPaDO0BpeFxZoMwFzpQUJsq0zADLUrxtEr9soq
// SIG // uIL/jFYIMW4hnbyp0LLp+lHY0Nb2uyeWd23rLYqTAkwu
// SIG // 3RAEN91BVqKKO7EYjkQtHUEj2kE3buQ0R5hPcHHdXr2Y
// SIG // FGlDrJyhggyhMIIMnQYKKwYBBAGCNwMDATGCDI0wggyJ
// SIG // BgkqhkiG9w0BBwKgggx6MIIMdgIBAzEPMA0GCWCGSAFl
// SIG // AwQCAQUAMIHdBgsqhkiG9w0BCRABBKCBzQSByjCBxwIB
// SIG // AQYJKwYBBAGgMgIDMDEwDQYJYIZIAWUDBAIBBQAEIOrk
// SIG // KOn64oIkvbQJDBwdn2AqvwQKo1FjnmUXuo596ZdBAhQ/
// SIG // 1upAigrHZWS6Y8TRPj2So6PozhgPMjAxNjAyMDcwNDM3
// SIG // NDRaoF2kWzBZMQswCQYDVQQGEwJTRzEfMB0GA1UEChMW
// SIG // R01PIEdsb2JhbFNpZ24gUHRlIEx0ZDEpMCcGA1UEAxMg
// SIG // R2xvYmFsU2lnbiBUU0EgZm9yIEFkdmFuY2VkIC0gRzKg
// SIG // ggjGMIIEqTCCA5GgAwIBAgISESEWwAmY3MaPon0lw4Y2
// SIG // qIO7MA0GCSqGSIb3DQEBCwUAMFsxCzAJBgNVBAYTAkJF
// SIG // MRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNhMTEwLwYD
// SIG // VQQDEyhHbG9iYWxTaWduIFRpbWVzdGFtcGluZyBDQSAt
// SIG // IFNIQTI1NiAtIEcyMB4XDTE1MDIwMzAwMDAwMFoXDTI2
// SIG // MDMwMzAwMDAwMFowWTELMAkGA1UEBhMCU0cxHzAdBgNV
// SIG // BAoTFkdNTyBHbG9iYWxTaWduIFB0ZSBMdGQxKTAnBgNV
// SIG // BAMTIEdsb2JhbFNpZ24gVFNBIGZvciBBZHZhbmNlZCAt
// SIG // IEcyMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKC
// SIG // AQEAt8eSiijY85HLUA1d6ylEwYJZpcpkzr2DtVHllMxu
// SIG // gJoXbNz/YuY2Q14Bwgxkjcp3nvZdd0W5Xsgk60E3EWY3
// SIG // rMjNdJvMs+9M4n8bznO+yZadNR5891T3bOygEtaumZ1j
// SIG // xHhfWakuIvXDJ9iyW0VjJ5D/jLE9qbTzDsDxIGUGiZ0D
// SIG // HKOwUkGFNOu8y85U9hL+swGJSqb6VMUf+9McwOO/l+M/
// SIG // 0Bv5QMcb1esmfEMwx3DyeBGqhU6lDg0ASjoRd6PKIwGA
// SIG // nsiTVl1GzIrY0qoKhIWyiVvVMs5BexhOVMqr0VJeEtMk
// SIG // nlbzDI/dvZKOedVBicOjoOWR/MdnnYczznEycwIDAQAB
// SIG // o4IBZzCCAWMwDgYDVR0PAQH/BAQDAgeAMEwGA1UdIARF
// SIG // MEMwQQYJKwYBBAGgMgEeMDQwMgYIKwYBBQUHAgEWJmh0
// SIG // dHBzOi8vd3d3Lmdsb2JhbHNpZ24uY29tL3JlcG9zaXRv
// SIG // cnkvMAkGA1UdEwQCMAAwFgYDVR0lAQH/BAwwCgYIKwYB
// SIG // BQUHAwgwRgYDVR0fBD8wPTA7oDmgN4Y1aHR0cDovL2Ny
// SIG // bC5nbG9iYWxzaWduLmNvbS9ncy9nc3RpbWVzdGFtcGlu
// SIG // Z3NoYTJnMi5jcmwwWAYIKwYBBQUHAQEETDBKMEgGCCsG
// SIG // AQUFBzAChjxodHRwOi8vc2VjdXJlLmdsb2JhbHNpZ24u
// SIG // Y29tL2NhY2VydC9nc3RpbWVzdGFtcGluZ3NoYTJnMi5j
// SIG // cnQwHQYDVR0OBBYEFC1ubtGN5QOA7udj6afZ2gs8VyI9
// SIG // MB8GA1UdIwQYMBaAFJIhp0qVXWSwm7Qe5gA3R+adQStM
// SIG // MA0GCSqGSIb3DQEBCwUAA4IBAQB/CgOhVXb+qvoU0RF6
// SIG // 4uIPOjL/e0bGBHGMXtDMeUfddQdtuki+P2DiyxEgwrar
// SIG // Q8e3otm4Wv7NtDqz8VJTDLSO/2LdiJuyGn8bUbdxh9pD
// SIG // SzdSzf4+BilwpCds59w8f3bjkMga6ZkSUO7qTCMayF+c
// SIG // 1SmEKuaBYgZ2gnajE1UOSKZR0fNMDTwIICiGaoO3nQyg
// SIG // XmA0BjGofXGPUfIE36Imk7nVKHejoLdCo4swJ1oJ9tyD
// SIG // mCHVfUGaBxkUWih2fa2goyQBTuEWyL9/R/M6BF5lPqJM
// SIG // //1vBs+suBsAZ/rva37X4gq/e8WnEqzXNPVXiD6Elq1r
// SIG // VC49Jw/iORrBxxhyMIIEFTCCAv2gAwIBAgILBAAAAAAB
// SIG // MYnGUAQwDQYJKoZIhvcNAQELBQAwTDEgMB4GA1UECxMX
// SIG // R2xvYmFsU2lnbiBSb290IENBIC0gUjMxEzARBgNVBAoT
// SIG // Ckdsb2JhbFNpZ24xEzARBgNVBAMTCkdsb2JhbFNpZ24w
// SIG // HhcNMTEwODAyMTAwMDAwWhcNMjkwMzI5MTAwMDAwWjBb
// SIG // MQswCQYDVQQGEwJCRTEZMBcGA1UEChMQR2xvYmFsU2ln
// SIG // biBudi1zYTExMC8GA1UEAxMoR2xvYmFsU2lnbiBUaW1l
// SIG // c3RhbXBpbmcgQ0EgLSBTSEEyNTYgLSBHMjCCASIwDQYJ
// SIG // KoZIhvcNAQEBBQADggEPADCCAQoCggEBAKqbjsOrEVEl
// SIG // AbaWlOJP2MEI9kYj2UXFlZdbqxq/0mxXyTMGH6APxjx+
// SIG // U0h6v52Hnq/uw4xH4ULs4+OhSmwMF8SmwbnNW/EeRImO
// SIG // /gveIVgT7k3IxWcLHLKz8TR2kaLLB203xaBHJgIVpJCR
// SIG // qXme1+tXnSt8ItgU1/EHHngiNmt3ea+v+X+OTuG1CDH9
// SIG // 6u1LcWKMI/EDOY9EebZ2A1eerS8IRtzSjLz0jnTOyGhp
// SIG // UXYRiw9dJFsZVD0mzECNgicbWSB9WfaTgI74Kjj9a6BA
// SIG // ZR9XdsxbjgRPLKjbhFATT8bci7n43WlMiOucezAm/HpY
// SIG // u1m8FHKSgVe3dsnYgAqAbgkCAwEAAaOB6DCB5TAOBgNV
// SIG // HQ8BAf8EBAMCAQYwEgYDVR0TAQH/BAgwBgEB/wIBADAd
// SIG // BgNVHQ4EFgQUkiGnSpVdZLCbtB7mADdH5p1BK0wwRwYD
// SIG // VR0gBEAwPjA8BgRVHSAAMDQwMgYIKwYBBQUHAgEWJmh0
// SIG // dHBzOi8vd3d3Lmdsb2JhbHNpZ24uY29tL3JlcG9zaXRv
// SIG // cnkvMDYGA1UdHwQvMC0wK6ApoCeGJWh0dHA6Ly9jcmwu
// SIG // Z2xvYmFsc2lnbi5uZXQvcm9vdC1yMy5jcmwwHwYDVR0j
// SIG // BBgwFoAUj/BLf6guRSSuTVD6Y5qL3uLdG7wwDQYJKoZI
// SIG // hvcNAQELBQADggEBAARWgkp80M7JvzZm0b41npNsl+gG
// SIG // zjEYWflsQV+ALsBCJbgYx/zUsTfEaKDPKGoDdEtjl4V3
// SIG // YTvXL+P1vTOikn0RH56KbO8ssPRijTZz0RY28bxe7LSA
// SIG // mHj80nZ56OEhlOAfxKLhqmfbs5xz5UAizznO2+Z3lae7
// SIG // ssv2GYadn8jUmAWycW9Oda7xPWRqO15ORqYqXQiS8aPz
// SIG // HXS/Yg0jjFwqOJXSwNXNz4jaHyi1uoFpZCq1pqLVc6/c
// SIG // RtsErpHXbsWYutRHxFZ0gEd4WIy+7yv97Gy/0ZT3v1Dg
// SIG // e+CQ/SAYeBgiXQgujBygl/MdmX2jnZHTBkROBG56HCDj
// SIG // NvC2ULkxggK0MIICsAIBATBxMFsxCzAJBgNVBAYTAkJF
// SIG // MRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNhMTEwLwYD
// SIG // VQQDEyhHbG9iYWxTaWduIFRpbWVzdGFtcGluZyBDQSAt
// SIG // IFNIQTI1NiAtIEcyAhIRIRbACZjcxo+ifSXDhjaog7sw
// SIG // DQYJYIZIAWUDBAIBBQCgggEUMBoGCSqGSIb3DQEJAzEN
// SIG // BgsqhkiG9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcNMTYw
// SIG // MjA3MDQzNzQ0WjAvBgkqhkiG9w0BCQQxIgQgYpgL67aT
// SIG // LNB+NjeP3KUYsks8HptKiceFZK8nYDeM16kwgaYGCyqG
// SIG // SIb3DQEJEAIMMYGWMIGTMIGQMIGNBBRXqt6jTjqEJxGX
// SIG // sll4jXMMauIuyTB1MF+kXTBbMQswCQYDVQQGEwJCRTEZ
// SIG // MBcGA1UEChMQR2xvYmFsU2lnbiBudi1zYTExMC8GA1UE
// SIG // AxMoR2xvYmFsU2lnbiBUaW1lc3RhbXBpbmcgQ0EgLSBT
// SIG // SEEyNTYgLSBHMgISESEWwAmY3MaPon0lw4Y2qIO7MA0G
// SIG // CSqGSIb3DQEBAQUABIIBAEQ1oC1dG0eyqeXTKg+7VOlF
// SIG // HSRU5GC0ezqLMFhoMMussiDxwJEn7k1AFnzo8krcLssX
// SIG // tDQ6bIzAxuslS0nsb+5TuGTKW0yWT2fWAIGhSYRBnopt
// SIG // fLQbejuTsBZeuhEkpLETWsE8jXK18k21K88brwA2PdqA
// SIG // W8pq6DQnh+Lbmyp2HXffy3pyXCIYgSJk9KIum1n870e1
// SIG // T/Xgar14qQDSHREeTesvjBj52L2E/tDlR5ETXk/gnMdU
// SIG // mmmujmATFNIX4wJljMja+jEiczScWfK4J2rjr+ADl9Vw
// SIG // HVv7HAGNNVxlhrgoQcXU5Auyj/B/EYz6guaJJebAj6DV
// SIG // wSrSnKp2xck=
// SIG // End signature block
